"""Assignments page for viewing, updating and creating task assignments."""

from __future__ import annotations

import datetime
from typing import Any, Dict, List

import streamlit as st
import pandas as pd

# Local services
from services.assignment_service import AssignmentService
from services.employee_service import EmployeeService
from services.task_service import TaskService


def _get_all(repo: Any) -> List[Dict[str, Any]]:
    """
    Safe fetch for TinyDB-backed repos with varying method names.
    Tries list_all() -> get_all() -> all() -> table.all().
    """
    for name in ("list_all", "get_all", "all"):
        if hasattr(repo, name):
            rows = getattr(repo, name)()
            return rows if isinstance(rows, list) else list(rows)
    if hasattr(repo, "table"):
        return [dict(r) for r in repo.table.all()]
    return []


def render() -> None:
    st.title("Assignments")

    role = str(st.session_state.get("role", ""))  # "Admin" | "Manager" | "Staff"
    can_edit = role in ("Admin", "Manager")

    assignment_service = AssignmentService()
    employee_service = EmployeeService()
    task_service = TaskService()

    # --- Load data ---
    employees = employee_service.list_employees(active_only=True)
    tasks = _get_all(task_service.repo)
    assignments = _get_all(assignment_service.repo)

    emp_df = pd.DataFrame(employees) if employees else pd.DataFrame(columns=["id", "full_name", "email"])
    task_df = pd.DataFrame(tasks) if tasks else pd.DataFrame(columns=["id", "title", "client_name", "due_date"])
    asg_df = pd.DataFrame(assignments) if assignments else pd.DataFrame(
        columns=[
            "id",
            "task_id",
            "employee_id",
            "assigned_on",
            "assigned_by",
            "status",
            "percent_complete",
            "remarks",
            "actual_completion_date",
        ]
    )

    # --- Create new assignment(s) ---
    st.subheader("Assign Tasks")
    if not can_edit:
        st.info("You have view-only access. Contact Admin/Manager to assign tasks.")
    else:
        if task_df.empty or emp_df.empty:
            st.warning("Add at least one Task and one Active Employee to create assignments.")
        else:
            # Task selector
            task_map = {f"{row['title']} (ID:{row['id']})": int(row["id"]) for _, row in task_df.iterrows()}
            sel_task_label = st.selectbox("Task", options=list(task_map.keys()), key="assign_task_sel")
            sel_task_id = task_map[sel_task_label]

            # Employees multi-select
            emp_map = {f"{row['full_name']} (ID:{row['id']})": int(row["id"]) for _, row in emp_df.iterrows()}
            sel_emp_labels = st.multiselect("Assign to Employees", options=list(emp_map.keys()), key="assign_emp_multi")
            sel_emp_ids = [emp_map[lbl] for lbl in sel_emp_labels]

            col1, col2, col3 = st.columns(3)
            with col1:
                assigned_on = st.date_input(
                    "Assigned On", value=datetime.date.today(), format="YYYY-MM-DD", key="assign_assigned_on"
                )
            with col2:
                status = st.selectbox(
                    "Status", options=["Pending", "In Progress", "Completed", "Deferred"], index=0, key="assign_status"
                )
            with col3:
                percent_complete = st.number_input(
                    "Percent Complete", min_value=0, max_value=100, value=0, step=5, key="assign_percent"
                )

            remarks = st.text_area("Remarks", value="", key="assign_remarks")
            assigned_by = st.session_state.get("username") or "system"

            if st.button("Create Assignments", key="assign_create_btn"):
                try:
                    if not sel_emp_ids:
                        raise ValueError("Please select at least one employee.")
                    for emp_id in sel_emp_ids:
                        payload = {
                            "task_id": int(sel_task_id),
                            "employee_id": int(emp_id),
                            "assigned_on": assigned_on,  # repo serializes to ISO
                            "assigned_by": assigned_by,
                            "status": status,
                            "percent_complete": int(percent_complete),
                            "remarks": remarks.strip(),
                            "actual_completion_date": None,
                        }
                        assignment_service.repo.create_assignment(payload)
                    st.success("Assignment(s) created.")
                    st.rerun()
                except Exception as e:
                    st.error(f"Unable to create assignments: {e}")

    # --- Existing assignments table (merged for readability) ---
    st.subheader("All Assignments")
    if asg_df.empty:
        st.info("No assignments yet.")
        return

    # Merge to show names
    show_df = asg_df.merge(
        emp_df[["id", "full_name"]], left_on="employee_id", right_on="id", how="left", suffixes=("", "_emp")
    )
    show_df = show_df.merge(
        task_df[["id", "title"]], left_on="task_id", right_on="id", how="left", suffixes=("", "_task")
    )
    show_df.rename(columns={"full_name": "employee", "title": "task"}, inplace=True)
    for col in ("assigned_on", "actual_completion_date"):
        if col in show_df.columns:
            show_df[col] = pd.to_datetime(show_df[col], errors="coerce").dt.date.astype("object")

    keep_cols = [
        "id",
        "employee",
        "task",
        "status",
        "percent_complete",
        "assigned_on",
        "actual_completion_date",
        "remarks",
    ]
    st.dataframe(show_df[keep_cols], use_container_width=True)

    # --- Quick update form ---
    st.subheader("Update Assignment Status")
    id_options = [int(x) for x in show_df["id"].dropna().tolist()]
    if not id_options:
        st.info("No selectable assignments.")
        return

    col_a, col_b, col_c = st.columns(3)
    with col_a:
        sel_id = st.selectbox("Assignment ID", options=id_options, key="upd_sel_id")
    with col_b:
        new_status = st.selectbox(
            "New Status", options=["Pending", "In Progress", "Completed", "Deferred"], key=f"upd_status_{sel_id}"
        )
    with col_c:
        new_percent = st.number_input(
            "Percent Complete", min_value=0, max_value=100, value=0, step=5, key=f"upd_percent_{sel_id}"
        )

    remarks_update = st.text_input("Remarks (optional)", value="", key=f"upd_remarks_{sel_id}")
    comp_date = None
    if new_status == "Completed":
        comp_date = st.date_input(
            "Actual Completion Date", value=datetime.date.today(), format="YYYY-MM-DD", key=f"upd_compdate_{sel_id}"
        )

    if st.button("Apply Update", key="upd_apply_btn"):
        try:
            fields: Dict[str, Any] = {
                "status": new_status,
                "percent_complete": int(new_percent),
                "remarks": remarks_update.strip(),
            }
            if comp_date:
                fields["actual_completion_date"] = comp_date  # repo serializes to ISO
            ok = assignment_service.repo.update_assignment(int(sel_id), **fields)
            if ok:
                st.success("Assignment updated.")
                st.rerun()
            else:
                st.warning("No changes were applied.")
        except Exception as e:
            st.error(f"Unable to update: {e}")
